/*
 * $Id$
 */
package tecgraf.ftc_1_3.server.states.v1_1;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_3.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_3.server.Session;
import tecgraf.ftc_1_3.server.states.State;

/**
 * Operao de obteno do tamanho do arquivo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class GetSizeState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que o tamanho do arquivo j foi lido.
     */
    SIZE_READ,
    /**
     * Estado que indica que o tamanho do arquivo j foi enviado.
     */
    SIZE_WRITTEN;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState;

  /**
   * O tamanho do arquivo.
   */
  private long size;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Cria a operao de obteno do tamanho do arquivo.
   */
  public GetSizeState() {
    this.currentState = InternalState.INITIAL;

    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado de obteno do tamanho do arquivo.");
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case INITIAL:
        //TODO: retornar codigo de retorno ok.
        FileChannel fileChannel = session.getFileChannel();
        try {
          this.size = fileChannel.size();
        }
        catch (IOException e) {
          this.size = -1;
          session.getFileServer().exceptionRaised(e,
            session.getFileChannelInfo().getFileId());
        }
        this.currentState = InternalState.SIZE_READ;
      case SIZE_READ:

        if (logger.isLoggable(Level.FINER))
          logger.finer("Enviando o tamanho do arquivo " + this.size);

        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        buffer.putLong(this.size);
        buffer.flip();
        try {
          if (channel.write(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          session.getFileServer().exceptionRaised(e,
            session.getFileChannelInfo().getFileId());
          return false;
        }
        finally {
          buffer.clear();
        }
        this.currentState = InternalState.SIZE_WRITTEN;
        session.setCurrentState(new GetOperationState());
      default:
        return true;
    }
  }
}
