/*
 * Detalhes da ltima alterao:
 * 
 * $Author: vfusco $ $Date: 2009-02-18 18:33:43 -0300 (Wed, 18 Feb 2009) $
 * $Revision: 88595 $
 */
package tecgraf.ftc_1_3.server.states.v1_1;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_3.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_3.server.FileChannelRequestInfo;
import tecgraf.ftc_1_3.server.FileServer;
import tecgraf.ftc_1_3.server.Session;
import tecgraf.ftc_1_3.server.states.State;
import tecgraf.ftc_1_3.utils.IOUtils;

/**
 * Operao para leitura de dados a partir de uma determinada posio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ReadState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que a posio a partir da qual os dados sero lidos j
     * foi lida.
     */
    POSITION_READ,
    /**
     * Estado que indica que a quantidade de bytes que sero lidos j foi lida.
     */
    BYTE_COUNT_READ,
    /**
     * Estado que indica que todos os bytes solicitados j foram enviados.
     */
    BYTES_SENT;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState;

  /**
   * A posio a partir da qual os dados sero lidos do arquivo.
   */
  private long position;
  /**
   * Quantidade de bytes que sero lidos do arquivo.
   */
  private long count;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Tamanho maximo a ser transmitido na chamada do metodo transferTo do
   * fileChannel.
   */
  public static final long MAX_BYTES = Integer.MAX_VALUE;

  /**
   * A quantidade de bytes enviados ao cliente.
   */
  private long bytesSent = 0;

  /**
   * Cria a operao para leitura de dados a partir de uma determinada posio.
   */
  public ReadState() {
    this.currentState = InternalState.INITIAL;

    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado de leitura.");
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    FileChannel fileChannel = session.getFileChannel();

    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          e.printStackTrace();
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.position = buffer.getLong();
        buffer.clear();

        if (this.position < 0) {
          try {
            this.position = fileChannel.position();
          }
          catch (Exception e1) {
            if (logger.isLoggable(Level.FINER))
              logger.finer("Erro ao ler posio do arquivo.");
            return false;
          }
        }

        this.currentState = InternalState.POSITION_READ;

        if (logger.isLoggable(Level.FINER))
          logger.finer("Position " + this.position);
      case POSITION_READ:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          e.printStackTrace();
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.count = buffer.getLong();
        buffer.clear();
        this.currentState = InternalState.BYTE_COUNT_READ;

        if (logger.isLoggable(Level.FINER))
          logger.finer("Quantidade pedida " + this.count);
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case BYTE_COUNT_READ:
        ByteBuffer buffer = session.getBuffer();
        FileChannel fileChannel = session.getFileChannel();
        FileChannelRequestInfo fileInfo = session.getFileChannelInfo();
        long bytesWritten = 0;
        try {
          long bytesToread = count - bytesSent;

          if (fileInfo.useTransferTo()
            && ((count <= MAX_BYTES) || (!FileServer.PLATAFORM_HAS_TRANSFERTO_BUG))) {

            if (logger.isLoggable(Level.FINEST)) {
              logger.finest("Using TransferTo");
              logger.finest("count=" + this.count);
              logger.finest("bytesSent=" + this.bytesSent);
              logger.finest("position=" + this.position);
            }

            bytesWritten =
              fileChannel.transferTo(this.position + this.bytesSent,
                bytesToread, channel);
          }
          else {
            //bytesWritten =
            //  IOUtils.transferTo(fileChannel, this.position + this.bytesSent,
            //   bytesToread, channel, buffer);
            bytesWritten =
              IOUtils.transferToNonBlock(fileChannel, this.position
                + this.bytesSent, bytesToread, channel, buffer);
            buffer.clear();
          }

          if (bytesWritten > 0) {
            session.markLastActivity();
            fileChannel.position(this.position + bytesWritten);
          }

        }
        catch (IOException e) {
          buffer.clear();
          session.getFileServer().exceptionRaised(e,
            session.getFileChannelInfo().getFileId());
          return false;
        }

        if (logger.isLoggable(Level.FINER))
          logger.finer("Enviados " + bytesWritten);

        this.bytesSent += bytesWritten;
        if (this.bytesSent == this.count) {
          this.currentState = InternalState.BYTES_SENT;
          session.setCurrentState(new GetOperationState());
        }
        return true;
      default:
        return true;
    }
  }
}
