/*
 * Detalhes da ltima alterao:
 * 
 * $Author$ $Date$ $Revision$
 */
package tecgraf.ftc_1_3.utils;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;

import tecgraf.ftc_1_3.common.logic.PrimitiveTypeSize;

/**
 * Classe utilitria para o tratamento de <i>buffers</i> de bytes (
 * {@link ByteBuffer}).
 * 
 * <p>
 * Todos os mtodos desta classe so sncronos, ou seja, ficam travados at que
 * se obtenha a resposta necessria.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ByteBufferUtils {
  /**
   * L uma valor do tipo long.
   * 
   * @param buffer O buffer utilizado na obteno do valor.
   * @param channel O canal de onde ser lido o valor.
   * 
   * @return O valor solicitado.
   * 
   * @throws IOException Caso ocorra algum problema na leitura do valor.
   */
  public static long readLong(ByteBuffer buffer, ReadableByteChannel channel)
    throws IOException {
    buffer.limit(PrimitiveTypeSize.LONG.getSize());
    int nbytes = 0;
    try {

      do {
        nbytes = channel.read(buffer);
        if (nbytes == -1)
          break;
      } while (buffer.hasRemaining());

      if (buffer.hasRemaining())
        return -1;

      buffer.flip();

      return buffer.getLong();
    }
    finally {
      buffer.clear();
    }
  }

  /**
   * L uma valor do tipo byte.
   * 
   * @param buffer O buffer utilizado na obteno do valor.
   * @param channel O canal de onde ser lido o valor.
   * 
   * @return O valor solicitado.
   * 
   * @throws IOException Caso ocorra algum problema na leitura do valor.
   */
  public static byte readByte(ByteBuffer buffer, ReadableByteChannel channel)
    throws IOException {
    buffer.limit(PrimitiveTypeSize.BYTE.getSize());
    int nbytes = 0;
    try {

      do {
        nbytes = channel.read(buffer);
        if (nbytes == -1)
          break;
      } while (buffer.hasRemaining());

      if (buffer.hasRemaining())
        return -1;

      buffer.flip();

      return buffer.get();
    }
    finally {
      buffer.clear();
    }
  }

  /**
   * Escreve um valor do tipo byte no canal.
   * 
   * @param buffer O buffer utilizado para escrever o valor.
   * @param channel O canal para onde ser enviado o valor.
   * @param value O valor.
   * 
   * @throws IOException Caso ocorra algum problema ao escrever o valor.
   */
  public static void writeByte(ByteBuffer buffer, WritableByteChannel channel,
    byte value) throws IOException {
    writeByte(buffer, channel, 0, value);
  }

  /**
   * Escreve um valor do tipo byte no canal.
   * 
   * <p>
   *  possvel enviar valores que j tenham sido inseridos anteriormente no
   * buffer.
   * 
   * @param buffer O buffer utilizado para escrever o valor.
   * @param channel O canal para onde ser enviado o valor.
   * @param extraBytes Quantos bytes j foram inseridos no buffer anteriormente
   *        e que tambm devem ser enviados.
   * @param value O valor.
   * 
   * @throws IOException Caso ocorra algum problema ao escrever o valor.
   */
  public static void writeByte(ByteBuffer buffer, WritableByteChannel channel,
    int extraBytes, byte value) throws IOException {
    buffer.put(value);
    buffer.flip();
    try {
      int bytesWritten = 0;
      do {
        bytesWritten += channel.write(buffer);
      } while (bytesWritten < (PrimitiveTypeSize.BYTE.getSize() + extraBytes));
    }
    finally {
      buffer.clear();
    }
  }

  /**
   * Escreve um valor do tipo long no canal.
   * 
   * @param buffer O buffer utilizado para escrever o valor.
   * @param channel O canal para onde ser enviado o valor.
   * @param value O valor.
   * 
   * @throws IOException Caso ocorra algum problema ao escrever o valor.
   */
  public static void writeLong(ByteBuffer buffer, WritableByteChannel channel,
    long value) throws IOException {
    writeLong(buffer, channel, 0, value);
  }

  /**
   * Escreve um valor do tipo long no canal.
   * 
   * <p>
   *  possvel enviar valores que j tenham sido inseridos anteriormente no
   * buffer.
   * 
   * @param buffer O buffer utilizado para escrever o valor.
   * @param channel O canal para onde ser enviado o valor.
   * @param extraBytes Quantos bytes j foram inseridos no buffer anteriormente
   *        e que tambm devem ser enviados.
   * @param value O valor.
   * 
   * @throws IOException Caso ocorra algum problema ao escrever o valor.
   */
  public static void writeLong(ByteBuffer buffer, WritableByteChannel channel,
    int extraBytes, long value) throws IOException {
    buffer.putLong(value);
    buffer.flip();
    try {
      int bytesWritten = 0;
      do {
        bytesWritten += channel.write(buffer);
      } while (bytesWritten < (PrimitiveTypeSize.LONG.getSize() + extraBytes));
    }
    finally {
      buffer.clear();
    }
  }

  /**
   * Escreve um valor do tipo byte[] no canal.
   * 
   * @param buffer O buffer utilizado para escrever o valor.
   * @param channel O canal para onde ser enviado o valor.
   * @param value O valor.
   * 
   * @throws IOException Caso ocorra algum problema ao escrever o valor.
   */
  public static void writeBytes(ByteBuffer buffer, WritableByteChannel channel,
    byte[] value) throws IOException {
    writeBytes(buffer, channel, 0, value);
  }

  /**
   * Escreve um valor do tipo byte[] no canal.
   * 
   * <p>
   *  possvel enviar valores que j tenham sido inseridos anteriormente no
   * buffer.
   * 
   * @param buffer O buffer utilizado para escrever o valor.
   * @param channel O canal para onde ser enviado o valor.
   * @param extraBytes Quantos bytes j foram inseridos no buffer anteriormente
   *        e que tambm devem ser enviados.
   * @param value O valor.
   * 
   * @throws IOException Caso ocorra algum problema ao escrever o valor.
   */
  public static void writeBytes(ByteBuffer buffer, WritableByteChannel channel,
    int extraBytes, byte[] value) throws IOException {
    buffer.put((byte) value.length);
    buffer.put(value);
    buffer.flip();
    int bytesWritten = 0;
    try {
      do {
        bytesWritten += channel.write(buffer);
      } while (bytesWritten < (value.length + extraBytes));
    }
    finally {
      buffer.clear();
    }
  }
}
